<?php

namespace App\Http\Controllers;

use Illuminate\Routing\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use App\Models\LokasiKerja;


class AbsenController extends Controller
{
    public function __construct()
    {
        // 1. Terapkan 'auth' pada SEMUA metode di controller ini
        $this->middleware('auth');

        // ATAU

        // 2. Terapkan 'auth' hanya pada metode 'index' dan 'show'
        // $this->middleware('auth')->only(['index', 'show']);

        // ATAU

        // 3. Terapkan 'auth' pada SEMUA metode KECUALI 'guestIndex' dan 'about'
        // $this->middleware('auth')->except(['guestIndex', 'about']);
    }

    public function absenkaryawan_show()
    {
        $lokasi = LokasiKerja::first();

        $userId = Auth::id();
        $today = date('Y-m-d');

        $absensi = DB::table('absensi')
            ->where('user_id', $userId)
            ->where('tanggal', $today)
            ->first();

        // Tentukan status untuk ditampilkan di Blade
        if (!$absensi) {
            $status = [
                'text' => 'Anda belum melakukan absensi masuk',
                'class' => 'bg-secondary-subtle text-secondary-emphasis'
            ];
        } elseif ($absensi->jam_masuk && !$absensi->jam_keluar) {
            $status = [
                'text' => $absensi->status === 'terlambat' ? 'Anda terlambat hadir' : 'Anda sudah hadir',
                'class' => $absensi->status === 'terlambat' 
                    ? 'bg-warning-subtle text-warning-emphasis'
                    : 'bg-success-subtle text-success-emphasis'
            ];
        } else {
            $status = [
                'text' => 'Anda sudah menyelesaikan absensi hari ini',
                'class' => 'bg-success-subtle text-success-emphasis'
            ];
        }
        

        return view('absensisemua', compact('absensi', 'status','lokasi'));
    }


    public function absenkaryawan_store(Request $request)
    {
        try {
            $request->validate([
                'jenis' => 'required',
                'jam' => 'required',
                'tanggal' => 'required',
                'latitude' => 'required',
                'longitude' => 'required',
            ]);

            $userId = Auth::id();
            $jamMasukWajib  = '08:00';
            $jamPulangWajib = '14:00';

            // Cek apakah user sudah absen hari ini
            $todayAbsensi = DB::table('absensi')
            ->where('user_id', $userId)
            ->where('tanggal', $request->tanggal)
            ->first();

           // ================= ABSEN MASUK =================
        if ($request->jenis === 'masuk') {

            if ($todayAbsensi) {
                return redirect()->back()->with('error', 'Anda sudah melakukan absen masuk hari ini.');
            }

            $status = ($request->jam > $jamMasukWajib) ? 'terlambat' : 'hadir';

                DB::table('absensi')->insert([
                    'user_id' => $userId,
                    'tanggal' => $request->tanggal,
                    'jam_masuk' => $request->jam,
                    'jam_keluar' => null,
                    'status' => $status, // ← pakai hasil logika controller
                    'lokasi_latitude' => $request->latitude,
                    'lokasi_longitude' => $request->longitude,
                ]);
                

            return redirect()->back()->with('success', 'Absensi berhasil!');
        }

        // ================= ABSEN PULANG =================
        if ($request->jenis === 'pulang') {

            if (!$todayAbsensi) {
                return redirect()->back()->with('error', 'Anda belum absen masuk hari ini.');
            }

            if ($request->jam < $jamPulangWajib) {
                return redirect()->back()
                    ->with('error', 'Absen pulang hanya bisa dilakukan setelah jam 14:00.');
            }

            DB::table('absensi')
                ->where('absensi_id', $todayAbsensi->absensi_id)
                ->update([
                    'jam_keluar' => $request->jam,
                    'lokasi_latitude' => $request->latitude,
                    'lokasi_longitude' => $request->longitude,
                ]);
        }

        return redirect()->route('absenkaryawan.show')
            ->with('success', 'Absensi berhasil disimpan!');
    } catch (\Exception $e) {
        return redirect()->back()
            ->with('error', $e->getMessage());
    }
    
}

    /* ============================================================
        BAGIAN UNTUK ADMIN (KELOLA ABSENSI)
    ============================================================ */

    // List semua absensi
    public function dataabsensiadmin()
    {
        $tanggal_a = date('Y-m-01');
        $tanggal_b = date('Y-m-d');

        $absensi = DB::table('absensi')
            ->join('users', 'users.id', '=', 'absensi.user_id')
            ->select('absensi.*', 'users.name')
            ->whereBetween('absensi.tanggal', [$tanggal_a, $tanggal_b])
            ->get();

        $data = [
            'absensi' => $absensi,
            'tanggal_a' => $tanggal_a,
            'tanggal_b' => $tanggal_b
        ];

        return view('halamanabsensiadmin', $data);
    }

    public function dataabsensiadmin_filter(Request $request)
    {
        $tanggal_a = $request->tanggal_awal;
        $tanggal_b = $request->tanggal_akhir;

        $absensi = DB::table('absensi')
            ->join('users', 'users.id', '=', 'absensi.user_id')
            ->select('absensi.*', 'users.name')
            ->whereBetween('absensi.tanggal', [$tanggal_a, $tanggal_b])
            ->get();

        $data = [
            'absensi' => $absensi,
            'tanggal_a' => $tanggal_a,
            'tanggal_b' => $tanggal_b
        ];

        return view('halamanabsensiadmin', $data);
    }


    public function absensilaporan(Request $request)
    {
        if ($request->isMethod('post')) {
            // === Jika POST, ambil data dari form ===
            $bulan = $request->bulan;
            $tahun = $request->tahun;
    
        } else {
            // === Jika GET, pakai tanggal sekarang ===
            $bulan = date('n');   // bulan berjalan
            $tahun = date('Y');   // tahun berjalan
        }

        // Jumlah hari dalam bulan ini
        $haripadabulan = cal_days_in_month(CAL_GREGORIAN, $bulan, $tahun);

        // Buat daftar hari + keterangan (Sabtu/Minggu)
        $listHari = [];

        for ($i = 1; $i <= $haripadabulan; $i++) {

            // Format tanggal YYYY-MM-DD
            $tanggal = sprintf('%04d-%02d-%02d', $tahun, $bulan, $i);

            // Buat objek Carbon
            $carbon = \Carbon\Carbon::parse($tanggal);

            // Nama hari dalam bahasa Indonesia
            $namaHari = $carbon->translatedFormat('l'); // Senin, Selasa, ...

            // Deteksi Sabtu/Minggu
            $keterangan = '';
            if ($namaHari == 'Sabtu') {
                $keterangan = 'Hari Sabtu';
            } elseif ($namaHari == 'Minggu') {
                $keterangan = 'Hari Minggu';
            }

            // Masukkan ke array
            $listHari[] = [
                'tanggal' => $tanggal,
                'hari' => $namaHari,
                'ket' => $keterangan
            ];
        }

        $tanggal_a = sprintf('%04d-%02d-01', $tahun, $bulan);
        // Tanggal terakhir bulan
        $tanggal_b = date("Y-m-t", strtotime($tanggal_a));

        $absensi = DB::table('absensi')
            ->join('users', 'users.id', '=', 'absensi.user_id')
            ->select('absensi.*', 'users.name')
            ->whereBetween('absensi.tanggal', [$tanggal_a, $tanggal_b])
            ->get();

        $karyawan = DB::table('users')->get();

        // Kirim data ke view
        $data = [
            'bulan' => $bulan,
            'tahun' => $tahun,
            'haripadabulan' => $haripadabulan,
            'listHari' => $listHari,
            'absensi' => $absensi,
            'karyawan' => $karyawan,
        ];

        return view('laporanbulanan', $data);
    }

    public function absenkaryawan2_show()
    {
        $userId = Auth::id();
        $today = date('Y-m-d');

        $absensi = DB::table('absensi')
            ->where('user_id', $userId)
            ->where('tanggal', $today)
            ->first();

        if (!$absensi) {
            $status = [
                'text' => 'Anda belum melakukan absensi masuk',
                'class' => 'bg-secondary-subtle text-secondary-emphasis'
            ];
        } elseif ($absensi->jam_masuk && !$absensi->jam_keluar) {
            $status = [
                'text' => 'Anda sudah absen masuk, silakan absen pulang',
                'class' => 'bg-warning-subtle text-warning-emphasis'
            ];
        } else {
            $status = [
                'text' => 'Anda sudah menyelesaikan absensi hari ini',
                'class' => 'bg-success-subtle text-success-emphasis'
            ];
        }

        return view('absensikaryawan', compact('absensi', 'status'));
    }

    
        public function cetakPdf(Request $request)
        {
            $bulan = $request->bulan;   // contoh: 11
            $tahun = $request->tahun;   // contoh: 2025
           

            // tanggal awal bulan
            $tanggal_a = Carbon::create($tahun, $bulan, 1)->startOfMonth()->toDateString();

            // tanggal akhir bulan
            $tanggal_b = Carbon::create($tahun, $bulan, 1)->endOfMonth()->toDateString();
            
            // $bulan = $request->bulan;
            $absensi = DB::table('absensi')
            ->join('users', 'users.id', '=', 'absensi.user_id')
            ->select(
                'absensi.user_id',
                'users.name',

                // total hadir (hadir + terlambat)
                DB::raw("SUM(CASE WHEN status IN ('hadir','terlambat') THEN 1 ELSE 0 END) AS total_hadir"),

                // total alpha
                DB::raw("SUM(CASE WHEN status = 'alpha' THEN 1 ELSE 0 END) AS total_alpha"),

                // absen lengkap = jam_masuk & jam_keluar tidak NULL
                DB::raw("SUM(CASE WHEN jam_masuk IS NOT NULL AND jam_keluar IS NOT NULL THEN 1 ELSE 0 END) AS absen_lengkap"),

                // absen tidak lengkap = salah satu NULL
                DB::raw("SUM(CASE WHEN jam_masuk IS NULL OR jam_keluar IS NULL THEN 1 ELSE 0 END) AS absen_tidak_lengkap")
            )
            ->whereBetween('absensi.tanggal', [$tanggal_a, $tanggal_b])
            ->groupBy('absensi.user_id', 'users.name')
            ->get();

            $data = [
                "laporan"   => $absensi,
                "bulan"     => $bulan,
                "tahun"     => $tahun,
                "totalharikerja" => $this->hitungHariKerja($bulan, $tahun),
            ];
            
            // === Generate PDF ===
            $pdf = Pdf::loadView('laporanprint', $data)->setPaper('A4', 'portrait');

            return $pdf->stream("Laporan-Absensi-$bulan-$tahun.pdf"); 

            // return view('laporanprint', $data);
        }


        private function hitungHariKerja($bulan, $tahun)
        {
            $start = Carbon::create($tahun, $bulan, 1);
            $end = $start->copy()->endOfMonth();

            $hariKerja = 0;

            for ($date = $start->copy(); $date->lte($end); $date->addDay()) {
                // hanya hitung Senin (1) s/d Jumat (5)
                if ($date->dayOfWeek >= 1 && $date->dayOfWeek <= 5) {
                    $hariKerja++;
                }
            }
            return $hariKerja;
        }

 
    public function edit()
    {
        $lokasi = DB::table('lokasi_kantor')->first();
        return view('absensisemua', compact('lokasiKantor'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'radius' => 'required|numeric',
        ]);

        DB::table('lokasi_kantor')->where('id', 1)->update([
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'radius' => $request->radius,
            'updated_at' => now()
        ]);

        return back()->with('success', 'Lokasi kantor berhasil diperbarui');
    }

 
    /**
     * Tampilkan halaman lokasi kerja
     */
    public function lokasikerja()
    {
        // Ambil 1 lokasi kerja (kantor)
        $lokasi = LokasiKerja::first();

        return view('lokasikerja', compact('lokasi'));
    }

    /**
     * Simpan / Update lokasi kerja
     */
    public function updatelokasikerja(Request $request)
    {
        $request->validate([
            'latitude'  => 'required|numeric',
            'longitude' => 'required|numeric',
            'radius'    => 'required|integer|min:10|max:1000',
        ]);

        // Ambil data pertama
        $lokasi = LokasiKerja::first();

        if (!$lokasi) {
            // Jika belum ada → insert
            LokasiKerja::create([
                'nama_lokasi' => 'Kantor Pusat',
                'latitude'    => $request->latitude,
                'longitude'   => $request->longitude,
                'radius'      => $request->radius,
            ]);
        } else {
            // Jika sudah ada → update
            $lokasi->update([
                'latitude'  => $request->latitude,
                'longitude' => $request->longitude,
                'radius'    => $request->radius,
            ]);
        }

        return redirect()->back()->with('success', 'Lokasi kerja berhasil disimpan.');
    }
}






        
