<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Aplikasi Absensi Interaktif</title>
    <!-- Bootstrap 5.3 CDN -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
    <!-- Font Inter -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; background-color: #f3f4f6; }
        .custom-card { border-radius: 1rem; box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -2px rgba(0, 0, 0, 0.06); }
        .btn { border-radius: 0.75rem; padding: 0.75rem 1.5rem; font-weight: 600; }
        .user-id-badge { background-color: #e9ecef; padding: 0.25rem 0.75rem; border-radius: 0.5rem; font-family: monospace; }
        .stat-box { border-radius: 0.75rem; padding: 1rem; }
    </style>
</head>
<body class="antialiased min-vh-100 d-flex flex-column align-items-center justify-content-start p-4">

    <div class="w-100" style="max-width: 900px;">
        <header class="text-center mb-4">
            <h1 class="h3 fw-bold text-dark">Sistem Absensi Digital (Geo-Fence)</h1>
            <p class="text-muted small mt-2">ID Pengguna: <span id="user-id-display" class="user-id-badge">Memuat...</span></p>
        </header>

        <!-- Status Dashboard -->
        <div class="card custom-card p-4 mb-4 border-top border-5 border-primary">
            <h2 class="h5 fw-semibold mb-3 text-secondary">Status Anda Saat Ini</h2>
            
            <div class="d-flex flex-column flex-md-row align-items-center justify-content-between">
                
                <!-- Waktu Server -->
                <div class="mb-4 mb-md-0 text-center text-md-start">
                    <p class="text-muted small mb-1">Waktu Server:</p>
                    <p id="current-time" class="fs-3 fw-bold text-dark">--:--:--</p>
                </div>

                <!-- Status Lokasi -->
                <div class="text-center my-3 my-md-0">
                    <p class="text-muted small mb-1">Status Lokasi:</p>
                    <p id="location-status" class="fs-5 fw-bold text-warning">Memuat Lokasi...</p>
                </div>

                <!-- Status Absensi -->
                <div class="text-center my-3 my-md-0">
                    <p class="text-muted small mb-1">Status Absensi:</p>
                    <p id="attendance-status" class="fs-4 fw-bold text-warning">Memuat...</p>
                </div>

                <!-- Action Buttons -->
                <div class="d-flex gap-3 mt-3 mt-md-0">
                    <button id="btn-masuk" class="btn btn-success" disabled>
                        Masuk
                    </button>
                    <button id="btn-keluar" class="btn btn-danger" disabled>
                        Keluar
                    </button>
                </div>
            </div>
             <p id="message-box" class="mt-3 text-center small fw-bold text-danger"></p>
        </div>

        <!-- Riwayat Absensi Hari Ini -->
        <div class="card custom-card p-4 mb-4">
            <h2 class="h5 fw-semibold mb-3 text-secondary">Riwayat Absensi Hari Ini</h2>
            <div class="table-responsive rounded border border-light-subtle">
                <table class="table table-striped table-sm mb-0">
                    <thead class="table-light">
                        <tr>
                            <th scope="col" class="px-3 py-2 text-start small text-muted text-uppercase">Waktu</th>
                            <th scope="col" class="px-3 py-2 text-start small text-muted text-uppercase">Jenis</th>
                            <th scope="col" class="px-3 py-2 text-start small text-muted text-uppercase">Lokasi</th>
                            <th scope="col" class="px-3 py-2 text-start small text-muted text-uppercase">Tanggal</th>
                        </tr>
                    </thead>
                    <tbody id="attendance-history-body">
                        <tr><td colspan="4" class="px-3 py-3 text-center text-muted">Memuat riwayat absensi...</td></tr>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- --- Laporan Absensi Lengkap (BARU) --- -->
        <div class="card custom-card p-4 mb-4">
            <h2 class="h5 fw-semibold mb-3 text-secondary">Laporan Absensi Lengkap</h2>
            
            <!-- Ringkasan Statistik -->
            <div class="row mb-4 g-3" id="report-summary">
                <div class="col-md-6">
                    <div class="stat-box bg-info-subtle text-info fw-bold">
                        <small class="d-block text-muted">Total Kehadiran Bulan Ini (Masuk/Keluar)</small>
                        <span class="fs-4" id="monthly-count">0</span> Pasang Absensi
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="stat-box bg-success-subtle text-success fw-bold">
                        <small class="d-block text-muted">Total Kehadiran Sepanjang Waktu</small>
                        <span class="fs-4" id="total-days-count">0</span> Hari
                    </div>
                </div>
            </div>

            <p class="text-muted small mt-4 mb-2">Semua Catatan Absensi (Diurutkan berdasarkan waktu terbaru):</p>
            
            <!-- Detail Laporan -->
            <div class="table-responsive rounded border border-light-subtle">
                <table class="table table-striped table-sm mb-0">
                    <thead class="table-dark">
                        <tr>
                            <th scope="col" class="px-3 py-2 text-start small text-white text-uppercase">Waktu Lengkap</th>
                            <th scope="col" class="px-3 py-2 text-start small text-white text-uppercase">Jenis</th>
                            <th scope="col" class="px-3 py-2 text-start small text-white text-uppercase">Tanggal</th>
                        </tr>
                    </thead>
                    <tbody id="detailed-report-body">
                        <tr><td colspan="3" class="px-3 py-3 text-center text-muted">Memuat semua riwayat absensi...</td></tr>
                    </tbody>
                </table>
            </div>
        </div>

    </div>

    <!-- Firebase SDKs -->
    <script type="module">
        import { initializeApp } from "https://www.gstatic.com/firebasejs/11.6.1/firebase-app.js";
        import { getAuth, signInAnonymously, signInWithCustomToken, onAuthStateChanged } from "https://www.gstatic.com/firebasejs/11.6.1/firebase-auth.js";
        import { getFirestore, collection, query, where, orderBy, onSnapshot, addDoc, serverTimestamp, setLogLevel, getDocs } from "https://www.gstatic.com/firebasejs/11.6.1/firebase-firestore.js";

        // setLogLevel('Debug'); // Aktifkan jika perlu debugging

        // Global Variables provided by the Canvas Environment
        const appId = typeof __app_id !== 'undefined' ? __app_id : 'default-absensi-app-id';
        const firebaseConfig = typeof __firebase_config !== 'undefined' ? JSON.parse(__firebase_config) : null;
        const initialAuthToken = typeof __initial_auth_token !== 'undefined' ? __initial_auth_token : null;

        // --- GEO FENCE CONFIGURATION (UBAH SESUAI LOKASI KANTOR ANDA) ---
        const TARGET_LAT = -6.1754; // Contoh: Monas, Jakarta Pusat
        const TARGET_LON = 106.8272; 
        const GEOFENCE_RADIUS_KM = 0.5; // Radius aman: 500 meter
        
        let db = null;
        let auth = null;
        let userId = null;
        let userStatus = 'Keluar'; 
        let isInsideGeofence = false;
        let userLocation = { lat: null, lon: null };

        // Elemen UI
        const attendanceStatusEl = document.getElementById('attendance-status');
        const userIdDisplayEl = document.getElementById('user-id-display');
        const historyBodyEl = document.getElementById('attendance-history-body');
        const detailedReportBodyEl = document.getElementById('detailed-report-body');
        const btnMasuk = document.getElementById('btn-masuk');
        const btnKeluar = document.getElementById('btn-keluar');
        const messageBox = document.getElementById('message-box');
        const currentTimeEl = document.getElementById('current-time');
        const locationStatusEl = document.getElementById('location-status');
        const monthlyCountEl = document.getElementById('monthly-count');
        const totalDaysCountEl = document.getElementById('total-days-count');


        // --- Utility Functions ---

        // Tampilkan waktu saat ini
        function updateTime() {
            const now = new Date();
            const timeString = now.toLocaleTimeString('id-ID', { hour: '2-digit', minute: '2-digit', second: '2-digit' });
            currentTimeEl.textContent = timeString;
        }
        setInterval(updateTime, 1000);
        updateTime();

        // Dapatkan kunci tanggal hari ini (YYYY-MM-DD)
        function getTodayDateKey() {
            const today = new Date();
            const year = today.getFullYear();
            const month = String(today.getMonth() + 1).padStart(2, '0');
            const day = String(today.getDate()).padStart(2, '0');
            return `${year}-${month}-${day}`;
        }
        
        // Dapatkan kunci bulan ini (YYYY-MM)
        function getCurrentMonthKey() {
            const today = new Date();
            const year = today.getFullYear();
            const month = String(today.getMonth() + 1).padStart(2, '0');
            return `${year}-${month}`;
        }

        // Haversine formula (menghitung jarak)
        function calculateDistance(lat1, lon1, lat2, lon2) {
            const R = 6371; 
            const dLat = (lat2 - lat1) * (Math.PI / 180);
            const dLon = (lon2 - lon1) * (Math.PI / 180);
            const a = 
                Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                Math.cos(lat1 * (Math.PI / 180)) * Math.cos(lat2 * (Math.PI / 180)) * Math.sin(dLon / 2) * Math.sin(dLon / 2);
            const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
            return R * c; 
        }

        // Perbarui UI berdasarkan status absensi dan geofence
        function updateUI() {
            // 1. Perbarui Status Absensi
            attendanceStatusEl.textContent = userStatus === 'Masuk' ? 'SEDANG BEKERJA' : (userStatus === 'Keluar' ? 'SUDAH KELUAR' : 'BELUM MASUK HARI INI');
            attendanceStatusEl.className = `fs-4 fw-bold ${
                userStatus === 'Masuk' ? 'text-success' : 'text-danger'
            }`;

            // 2. Kontrol Tombol
            const canMasuk = userStatus !== 'Masuk' && isInsideGeofence;
            const canKeluar = userStatus === 'Masuk' && isInsideGeofence;
            
            btnMasuk.disabled = !canMasuk;
            btnKeluar.disabled = !canKeluar;
            
            if (isInsideGeofence) {
                messageBox.textContent = '';
            }
        }

        // --- Geolocation Functions ---

        function checkGeolocationAndStatus() {
            locationStatusEl.textContent = 'Mendapatkan lokasi...';
            locationStatusEl.classList.remove('text-success', 'text-danger');
            locationStatusEl.classList.add('text-warning');

            if (!navigator.geolocation) {
                locationStatusEl.textContent = 'Geolocation tidak didukung.';
                locationStatusEl.classList.replace('text-warning', 'text-danger');
                messageBox.textContent = 'Absensi dibatasi karena Geolocation tidak tersedia.';
                return;
            }

            navigator.geolocation.getCurrentPosition((position) => {
                const userLat = position.coords.latitude;
                const userLon = position.coords.longitude;
                userLocation = { lat: userLat, lon: userLon };

                const distance = calculateDistance(userLat, userLon, TARGET_LAT, TARGET_LON);
                
                isInsideGeofence = distance <= GEOFENCE_RADIUS_KM;
                const distanceInMeters = (distance * 1000).toFixed(0);
                
                if (isInsideGeofence) {
                    locationStatusEl.textContent = `DI DALAM JANGKAUAN (${distanceInMeters}m)`;
                    locationStatusEl.classList.replace('text-warning', 'text-success');
                } else {
                    locationStatusEl.textContent = `DI LUAR JANGKAUAN (${distanceInMeters}m)`;
                    locationStatusEl.classList.replace('text-warning', 'text-danger');
                    messageBox.textContent = `Anda berada ${distanceInMeters} meter dari lokasi kantor. Tidak dapat absensi.`;
                }

                updateUI();
                
            }, (error) => {
                locationStatusEl.textContent = 'Gagal mendapatkan lokasi (Izinkan akses).';
                locationStatusEl.classList.replace('text-warning', 'text-danger');
                messageBox.textContent = 'Izin lokasi dibutuhkan untuk absensi.';
                isInsideGeofence = false; 
                updateUI();
                console.error("Geolocation Error:", error.message);
            }, {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            });
        }

        // --- Core Firebase Functions ---

        async function initializeFirebase() {
            if (!firebaseConfig) {
                console.error("Firebase config is missing.");
                userIdDisplayEl.textContent = 'Error Config';
                return;
            }

            const app = initializeApp(firebaseConfig);
            auth = getAuth(app);
            db = getFirestore(app);

            try {
                if (initialAuthToken) {
                    await signInWithCustomToken(auth, initialAuthToken);
                } else {
                    await signInAnonymously(auth);
                }
            } catch (error) {
                console.error("Firebase Authentication failed:", error);
                await signInAnonymously(auth);
            }

            onAuthStateChanged(auth, (user) => {
                if (user) {
                    userId = user.uid;
                } else {
                    userId = crypto.randomUUID(); 
                }
                userIdDisplayEl.textContent = userId.substring(0, 8) + '...';
                
                if (db && userId) {
                    startRealtimeListener(); // Listener untuk hari ini
                    renderDetailedReport(); // Ambil semua data untuk laporan
                    
                    checkGeolocationAndStatus();
                    setInterval(checkGeolocationAndStatus, 15000); 
                }
            });
        }

        // Fungsi untuk mencatat absensi
        async function checkInOrOut(type) {
            if (!db || !userId) {
                messageBox.textContent = 'Sistem belum siap. Coba lagi.';
                return;
            }
            
            if (!isInsideGeofence) {
                messageBox.textContent = 'Gagal absensi. Anda berada di luar Geo-Fence kantor.';
                checkGeolocationAndStatus(); 
                return;
            }

            if (type === 'Masuk' && userStatus === 'Masuk') {
                messageBox.textContent = 'Anda sudah tercatat Masuk.';
                return;
            }
            if (type === 'Keluar' && userStatus !== 'Masuk') {
                 messageBox.textContent = 'Anda belum tercatat Masuk hari ini.';
                return;
            }

            try {
                const todayKey = getTodayDateKey();
                const colRef = collection(db, `artifacts/${appId}/public/data/attendance_records`);

                await addDoc(colRef, {
                    userId: userId,
                    type: type,
                    timestamp: serverTimestamp(),
                    dateKey: todayKey,
                    location: userLocation 
                });

                messageBox.textContent = `Absensi ${type} berhasil dicatat!`;
            } catch (e) {
                console.error("Error adding document: ", e);
                messageBox.textContent = `Gagal mencatat absensi: ${e.message}`;
            }
        }

        // Real-time listener untuk riwayat absensi hari ini (Table atas)
        function startRealtimeListener() {
            const todayKey = getTodayDateKey();
            const colRef = collection(db, `artifacts/${appId}/public/data/attendance_records`);
            
            const q = query(
                colRef,
                where('userId', '==', userId),
                where('dateKey', '==', todayKey),
                orderBy('timestamp', 'desc')
            );

            onSnapshot(q, (snapshot) => {
                const records = [];
                snapshot.forEach((doc) => {
                    const data = doc.data();
                    records.push({
                        ...data,
                        id: doc.id,
                        displayTime: data.timestamp ? data.timestamp.toDate().toLocaleTimeString('id-ID', { hour: '2-digit', minute: '2-digit', second: '2-digit' }) : 'Memuat...',
                        fullDateTime: data.timestamp ? data.timestamp.toDate().toLocaleString('id-ID') : 'Memuat...'
                    });
                });

                renderTodayHistory(records);

                let newStatus = 'Belum Masuk';
                if (records.length > 0) {
                    newStatus = records[0].type;
                }
                userStatus = newStatus; 
                updateUI();

            }, (error) => {
                console.error("Error listening to attendance records:", error);
                historyBodyEl.innerHTML = `<tr><td colspan="4" class="px-3 py-3 text-center text-danger">Gagal memuat riwayat: ${error.message}</td></tr>`;
            });
        }

        // Render riwayat absensi hari ini (Table atas)
        function renderTodayHistory(records) {
            historyBodyEl.innerHTML = '';
            if (records.length === 0) {
                historyBodyEl.innerHTML = `<tr><td colspan="4" class="px-3 py-3 text-center text-muted">Belum ada catatan absensi hari ini.</td></tr>`;
                return;
            }

            records.forEach(record => {
                const row = document.createElement('tr');
                row.className = record.type === 'Masuk' ? 'table-success' : 'table-danger';
                
                const typeColor = record.type === 'Masuk' ? 'text-success' : 'text-danger';
                
                const locationDisplay = record.location && record.location.lat 
                    ? `L:${record.location.lat.toFixed(4)}, B:${record.location.lon.toFixed(4)}`
                    : 'Tidak Tersedia';

                row.innerHTML = `
                    <td class="px-3 py-2 fw-medium text-dark">${record.displayTime}</td>
                    <td class="px-3 py-2 fw-semibold ${typeColor}">${record.type}</td>
                    <td class="px-3 py-2 small text-muted">${locationDisplay}</td>
                    <td class="px-3 py-2 text-muted">${record.dateKey}</td>
                `;
                historyBodyEl.appendChild(row);
            });
        }
        
        // --- Fungsi Laporan Detil (BARU) ---
        async function renderDetailedReport() {
            detailedReportBodyEl.innerHTML = `<tr><td colspan="3" class="px-3 py-3 text-center text-warning">Mengambil data laporan...</td></tr>`;

            if (!db || !userId) {
                detailedReportBodyEl.innerHTML = `<tr><td colspan="3" class="px-3 py-3 text-center text-danger">Database belum terinisialisasi.</td></tr>`;
                return;
            }

            try {
                const colRef = collection(db, `artifacts/${appId}/public/data/attendance_records`);
                
                // Ambil semua dokumen untuk user ini, diurutkan dari yang terbaru
                const q = query(
                    colRef,
                    where('userId', '==', userId),
                    orderBy('timestamp', 'desc')
                );

                const snapshot = await getDocs(q);
                const allRecords = [];
                
                // Variabel untuk statistik
                const monthlyAttendance = new Set();
                const allAttendanceDays = new Set();
                const currentMonthKey = getCurrentMonthKey();

                snapshot.forEach((doc) => {
                    const data = doc.data();
                    const timestampDate = data.timestamp ? data.timestamp.toDate() : new Date();
                    const fullDateTime = timestampDate.toLocaleString('id-ID', { year: 'numeric', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit', second: '2-digit' });
                    
                    allRecords.push({
                        ...data,
                        fullDateTime: fullDateTime,
                        dateKey: data.dateKey,
                        monthKey: data.dateKey ? data.dateKey.substring(0, 7) : '', // YYYY-MM
                    });

                    // Hitung statistik
                    if (data.type === 'Masuk') {
                        allAttendanceDays.add(data.dateKey);
                        if (data.dateKey.startsWith(currentMonthKey)) {
                            monthlyAttendance.add(data.dateKey);
                        }
                    }
                });
                
                // Update Statistik UI
                monthlyCountEl.textContent = monthlyAttendance.size;
                totalDaysCountEl.textContent = allAttendanceDays.size;


                // Render Tabel Laporan Detail
                detailedReportBodyEl.innerHTML = '';
                if (allRecords.length === 0) {
                    detailedReportBodyEl.innerHTML = `<tr><td colspan="3" class="px-3 py-3 text-center text-muted">Tidak ada riwayat absensi sama sekali.</td></tr>`;
                    return;
                }

                allRecords.forEach(record => {
                    const row = document.createElement('tr');
                    const typeClass = record.type === 'Masuk' ? 'text-success' : 'text-danger';
                    
                    row.innerHTML = `
                        <td class="px-3 py-2 small text-dark">${record.fullDateTime}</td>
                        <td class="px-3 py-2 fw-semibold ${typeClass}">${record.type}</td>
                        <td class="px-3 py-2 text-muted">${record.dateKey}</td>
                    `;
                    detailedReportBodyEl.appendChild(row);
                });

            } catch (error) {
                console.error("Gagal memuat laporan: ", error);
                detailedReportBodyEl.innerHTML = `<tr><td colspan="3" class="px-3 py-3 text-center text-danger">Error memuat data: ${error.message}</td></tr>`;
            }
        }

        // --- Event Listeners ---
        btnMasuk.addEventListener('click', () => checkInOrOut('Masuk'));
        btnKeluar.addEventListener('click', () => checkInOrOut('Keluar'));

        // Start initialization
        initializeFirebase();
    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>
</body>
</html>
<?php /**PATH C:\aplikasi-absensi1\aplikasi-absensi\resources\views/laporan.blade.php ENDPATH**/ ?>